﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

//Extensions is a static class containing extension methods for many types
public static class Extensions
{
    /// <summary>
    /// Finds a child of a game object by name
    /// </summary>
    /// <param name="obj">The parent object to check</param>
    /// <param name="name">The name of the child object to find</param>
    /// <returns>The child object if found, otherwise null</returns>
    public static GameObject FindChild(this GameObject obj, string name)
    {
        if (obj != null)
        {
            foreach (Transform child in obj.transform)
            {
                if (child.gameObject.name == name)
                {
                    return child.gameObject;
                }
            }
        }

        return null;
    }

    /// <summary>
    /// Gets all child transforms from a parent transform
    /// </summary>
    /// <param name="parent">The parent transform to check</param>
    /// <param name="transformList">A list of known transforms - do NOT use, for internal recursion only</param>
    /// <returns>A list of all child transforms</returns>
    public static List<Transform> GetAllChildren(this Transform parent, List<Transform> transformList = null)
    {
        if (transformList == null)
        {
            transformList = new List<Transform>();
        }

        foreach (Transform child in parent)
        {
            transformList.Add(child);
            child.GetAllChildren(transformList);
        }

        return transformList;
    }

    /// <summary>
    /// Enables the game object and all children by making them interactable
    /// </summary>
    /// <param name="obj">The game object to enable</param>
    public static void Enable(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Enable();
                }

                foreach (Component component in child.GetComponents(typeof(Selectable)))
                {
                    ((Selectable)component).interactable = true;
                }

                /*foreach (Component component in child.GetComponents(typeof(CustomButton)))
                {
                    ((CustomButton)component).Enable();
                }*/
            }
        }

        catch (Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Disables the game object and all children by making them uninteractable
    /// </summary>
    /// <param name="obj">The game object to disable</param>
    public static void Disable(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Disable();
                }

                foreach (Component component in child.GetComponents(typeof(Selectable)))
                {
                    ((Selectable)component).interactable = false;
                }

                /*foreach (Component component in child.GetComponents(typeof(CustomButton)))
                {
                    ((CustomButton)component).Disable();
                }*/
            }
        }

        catch (Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Randomly shuffles a list
    /// </summary>
    /// <typeparam name="T">The list type</typeparam>
    /// <param name="list">The list to shuffle</param>
    public static void Shuffle<T>(this IList<T> list)
    {
        System.Random rng = new System.Random();

        int n = list.Count;

        while (n > 1)
        {
            n--;
            int k = rng.Next(n + 1);
            T value = list[k];
            list[k] = list[n];
            list[n] = value;
        }
    }

    /// <summary>
    /// Extension for System.Random to return a random double between a min and max value
    /// </summary>
    /// <param name="random">The random object</param>
    /// <param name="minValue">The minimum value boundary</param>
    /// <param name="maxValue">The maximum value boundary</param>
    /// <returns></returns>
    public static double NextDouble(this System.Random random, double minValue, double maxValue)
    {
        return random.NextDouble() * (maxValue - minValue) + minValue;
    }

    /// <summary>
    /// Converts an integer into its string word equivalent
    /// </summary>
    /// <param name="number">The number to convert</param>
    /// <returns>The word representing the number</returns>
    public static string ToWordString(this int number)
    {
        if (number == 0)
        {
            return "zero";
        }

        if (number < 0)
        {
            return "minus " + ToWordString(Math.Abs(number));
        }

        string words = "";

        if ((number / 1000000) > 0)
        {
            words += ToWordString(number / 1000000) + " million ";
            number %= 1000000;
        }

        if ((number / 1000) > 0)
        {
            words += ToWordString(number / 1000) + " thousand ";
            number %= 1000;
        }

        if ((number / 100) > 0)
        {
            words += ToWordString(number / 100) + " hundred ";
            number %= 100;
        }

        if (number > 0)
        {
            if (words != "")
            {
                words += "and ";
            }

            var unitsMap = new[] { "zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine", "ten", "eleven", "twelve", "thirteen", "fourteen", "fifteen", "sixteen", "seventeen", "eighteen", "nineteen" };
            var tensMap = new[] { "zero", "ten", "twenty", "thirty", "forty", "fifty", "sixty", "seventy", "eighty", "ninety" };

            if (number < 20)
            {
                words += unitsMap[number];
            }

            else
            {
                words += tensMap[number / 10];

                if ((number % 10) > 0)
                {
                    words += "-" + unitsMap[number % 10];
                }
            }
        }

        return words;
    }

    /// <summary>
    /// Capitializes the first letter of a string
    /// </summary>
    /// <param name="str">The string to capitialize</param>
    /// <returns>The string with the first letter capitialized</returns>
    public static string CapitalizeFirstLetter(this string str)
    {
        if (str.Length > 1)
        {
            return char.ToUpper(str[0]) + str.Substring(1);
        }

        else
        {
            return str;
        }
    }

    /// <summary>
    /// Saves texture data to a PNG file
    /// </summary>
    /// <param name="texture">The texture to save</param>
    /// <param name="filePath">A path to a file to save the texture to</param>
    /// <param name="overwrite">Should the file be overwritten if it already exists?</param>
    public static void SaveToFile(this Texture2D texture, string filePath, bool overwrite = true)
    {
        //Apparently WriteAllBytes doesn't overwrite in Unity so...
        if(overwrite)
        {
            File.Delete(filePath);
        }

        File.WriteAllBytes(filePath, texture.EncodeToPNG());
    }

    /// <summary>
    /// Returns whether the string represents a float
    /// </summary>
    /// <param name="str">The string to parse</param>
    /// <returns>A boolean representing whether the string represents a float or not</returns>
    public static bool IsFloat(this string str)
    {
        float flo;
        return float.TryParse(str, out flo);
    }

    /// <summary>
    /// Tries to convert a comma separated string into a Vector2, returning the sentinel value if unsuccessful
    /// </summary>
    /// <param name="str">The string to convert</param>
    /// <returns>The Vector2 representation if successful, otherwise the sentinel value</returns>
    public static Vector2 ToVector2(this string str)
    {
        try
        {
            string[] values = str.Split(',');
            if(values.Count() >= 2)
            {
                if (values[0].IsFloat() && values[1].IsFloat())
                {
                    return new Vector2(Convert.ToSingle(values[0]), Convert.ToSingle(values[1]));
                }
            }

            return Constants.SentinelVector2;
        }

        catch
        {
            return Constants.SentinelVector2;
        }
    }
}